﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/opensearch/OpenSearchServiceRequest.h>
#include <aws/opensearch/OpenSearchService_EXPORTS.h>
#include <aws/opensearch/model/AppConfig.h>
#include <aws/opensearch/model/DataSource.h>

#include <utility>

namespace Aws {
namespace OpenSearchService {
namespace Model {

/**
 */
class UpdateApplicationRequest : public OpenSearchServiceRequest {
 public:
  AWS_OPENSEARCHSERVICE_API UpdateApplicationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateApplication"; }

  AWS_OPENSEARCHSERVICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique identifier for the OpenSearch application to be updated.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  UpdateApplicationRequest& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data sources to associate with the OpenSearch application.</p>
   */
  inline const Aws::Vector<DataSource>& GetDataSources() const { return m_dataSources; }
  inline bool DataSourcesHasBeenSet() const { return m_dataSourcesHasBeenSet; }
  template <typename DataSourcesT = Aws::Vector<DataSource>>
  void SetDataSources(DataSourcesT&& value) {
    m_dataSourcesHasBeenSet = true;
    m_dataSources = std::forward<DataSourcesT>(value);
  }
  template <typename DataSourcesT = Aws::Vector<DataSource>>
  UpdateApplicationRequest& WithDataSources(DataSourcesT&& value) {
    SetDataSources(std::forward<DataSourcesT>(value));
    return *this;
  }
  template <typename DataSourcesT = DataSource>
  UpdateApplicationRequest& AddDataSources(DataSourcesT&& value) {
    m_dataSourcesHasBeenSet = true;
    m_dataSources.emplace_back(std::forward<DataSourcesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration settings to modify for the OpenSearch application.</p>
   */
  inline const Aws::Vector<AppConfig>& GetAppConfigs() const { return m_appConfigs; }
  inline bool AppConfigsHasBeenSet() const { return m_appConfigsHasBeenSet; }
  template <typename AppConfigsT = Aws::Vector<AppConfig>>
  void SetAppConfigs(AppConfigsT&& value) {
    m_appConfigsHasBeenSet = true;
    m_appConfigs = std::forward<AppConfigsT>(value);
  }
  template <typename AppConfigsT = Aws::Vector<AppConfig>>
  UpdateApplicationRequest& WithAppConfigs(AppConfigsT&& value) {
    SetAppConfigs(std::forward<AppConfigsT>(value));
    return *this;
  }
  template <typename AppConfigsT = AppConfig>
  UpdateApplicationRequest& AddAppConfigs(AppConfigsT&& value) {
    m_appConfigsHasBeenSet = true;
    m_appConfigs.emplace_back(std::forward<AppConfigsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;

  Aws::Vector<DataSource> m_dataSources;

  Aws::Vector<AppConfig> m_appConfigs;
  bool m_idHasBeenSet = false;
  bool m_dataSourcesHasBeenSet = false;
  bool m_appConfigsHasBeenSet = false;
};

}  // namespace Model
}  // namespace OpenSearchService
}  // namespace Aws
